Library BdpiSeries;
Uses
  DbExpCA400, ClientAccess, DBXpress,
  FMTBcd,
  Classes, SysUtils, Windows;

{$R BdpiSeries.res} // version info

Type
  // alias so we have access to the protected methods
  TBdpConnection = Class(TCA400SQLConnection);
  TBdpCommand = Class(TCA400SQLCommand);
  TBdpCursor = Class(TCA400SQLCursor);
  TBdpMetadata = Class(TCA400SQLMetaData);
  TBdpMetadataCursor = Class(TCA400MetaDataCursor);

  THandleContainer = Class
  Private
    FList: TList;
    FLock: TRTLCriticalSection;
    Function Get(Index: Integer): TObject;
  Public
    Constructor Create;
    Destructor Destroy; Override;
    Function Add (Value: TObject): Integer;
    Procedure Remove (Value: Integer);
    Property Items[Index: Integer]: TObject Read Get; Default;
  End;

Var
  Connections: THandleContainer;
  Commands: THandleContainer;
  Cursors: THandleContainer;
  Metadata: THandleContainer;

Procedure Info (Value: String); Begin MessageBox(0, PChar(Value), 'Info', mb_Ok) End;

{ Helper }

Constructor THandleContainer.Create;
Begin
  Inherited Create;
  FList:= TList.Create;
  InitializeCriticalSection(FLock);
End;

Destructor THandleContainer.Destroy;
Begin
  FList.Free;
  DeleteCriticalSection(FLock);
  Inherited Destroy
End;

Function THandleContainer.Add (Value: TObject): Integer;
Var
  i: Integer;
Begin
  EnterCriticalSection(FLock);
  Try
    For i:= 0 To FList.Count-1 Do
      If FList[i] = Nil Then Begin
        FList[i]:= Pointer(Value);
        Result:= i;
        Exit
      End;
    Result:= FList.Add(Pointer(Value))
  Finally
    LeaveCriticalSection(FLock)
  End
End;

Procedure THandleContainer.Remove (Value: Integer);
Begin
  EnterCriticalSection(FLock);
  Try
    If (Value>=0) And (Value<FList.Count) Then
      FList[Value]:= Nil
  Finally
    LeaveCriticalSection(FLock)
  End
End;

Function THandleContainer.Get(Index: Integer): TObject;
Begin
  If (Index>=0) And (Index<FList.Count) Then
    Result:= TObject(FList[Index])
  Else
    Result:= Nil
End;

{ conversion }

Function DbxType(Value: Word): Word;
Begin
  Case Value Of
    0: Result:= fldUNKNOWN;
    1: Result:= fldZSTRING;    // String
    2: Result:= fldBOOL;       // Boolean
    3: Result:= fldINT16;      // Int16
    4: Result:= fldINT32;      // Int32
    5: Result:= fldINT64;      // Int64
    6: Result:= fldFLOATIEEE;  // Float
    7: Result:= fldFLOAT;      // Double
    8: Result:= fldFMTBCD;     // Decimal
    9: Result:= fldDATE;       // Date
   10: Result:= fldTIME;       // Time
   11: Result:= fldTIMESTAMP;  // DateTime
   12: Result:= fldBYTES;      // Bytes
   13: Result:= fldVARBYTES;   // VarBytes
   14: Result:= fldBLOB;       // Blob
   15: Result:= fldCURSOR;     // Cursor
   16: Result:= fldUNKNOWN;    // Guid
   17: Result:= fldADT;        // Adt
   18: Result:= fldARRAY;      // Array
   19: Result:= fldREF;        // Ref
   20: Result:= fldTABLE;      // Nested
 Else
   Result:= fldUNKNOWN
 End
End;

Function DbxSubType(Value: Word): Word;
Begin
  Case Value Of
    51: Result:= fldstFIXED;
    52: Result:= fldstBINARY;
    53: Result:= fldstMEMO;
    54: Result:= fldstHBINARY;
    55: Result:= fldstHMEMO;
    56: Result:= fldstBFILE;
  Else
    Result:= 0
  End
End;

Function BdpType(Value: Word): Word;
Const
  MapType: Array[fldUNKNOWN..fldFMTBCD] Of Word = (0, 1, 9, 14, 2, 3, 4, 6, 8, 12, 10, 11, 3, 4, 6, 13, 0, 15, 5, 5, 17, 18, 19, 20, 11, 8);
Begin
  If {Value>=fldUNKNOWN And } (Value<=fldFMTBCD) Then
    Result:= MapType[Value]
  Else
    Result:= 0
End;

Function BdpSubType(Value: Word): Word;
Begin
  Case Value Of
    fldstFIXED:   Result:= 51; // stFixed
    fldstBINARY:  Result:= 52; // stBinary
    fldstMEMO:    Result:= 53; // stMemo
    fldstHBINARY: Result:= 54; // stHBinary
    fldstHMEMO:   Result:= 55; // stHMemo
    fldstBFILE:   Result:= 56; // stBFile
  Else
    Result:= 0
  End
End;

{ Trace }


Function Trace(const Value: String): Boolean; Overload;
Begin
  Result:= True;
  OutputDebugString(PChar('bdpiSeries: '+Value));
end;

Function Trace(const Value: String; Items: array of Const): Boolean; Overload;
Begin
  Result:= Trace(Format(Value, Items))
End;


{ BdpConnection related }

Function ConInit (szVendorLib: PChar; szResource: PChar): Integer; StdCall;
Begin
  If BindError Then
    Result:= DBXERR_INVALIDHNDL
  Else
    Result:= SQL_SUCCESS
End;

Function ConExit: Integer; StdCall;
Begin Try
  Result:= SQL_SUCCESS
Except
  Result:= DBXERR_INVALIDHNDL
End End;

Function ConAlloc(Var Handle: Integer): Integer; StdCall;
Begin Try
  Handle:= Connections.Add(TBdpConnection.Create(Nil));
  Assert(Trace('ConAlloc %d', [Handle]));
  Result:= SQL_SUCCESS
Except
  Result:= DBXERR_INVALIDHNDL
End End;

Function ConFree(Handle: Integer): Integer; StdCall;
Begin Try
  Assert(Trace('ConFree %d', [Handle]));
  TBdpConnection(Connections[Handle]).Free;
  Connections.Remove(Handle);
  Result:= SQL_SUCCESS
Except
  Result:= DBXERR_INVALIDHNDL
End End;

Function ConConnect(Handle: Integer; szDatabase: PChar; szUser: PChar; szPasswd: PChar): Integer; StdCall;
Begin Try
  Assert(Trace('ConConnect %d', [Handle]));
  Result:= TBdpConnection(Connections[Handle]).connect(szDatabase, szUser, szPasswd)
Except
  Result:= DBXERR_INVALIDHNDL
End End;

Function ConDisConnect(Handle: Integer): Integer; StdCall;
Begin Try
  Assert(Trace('ConDisConnect %d', [Handle]));
  Result:= TBdpConnection(Connections[Handle]).Disconnect
Except
  Result:= DBXERR_INVALIDHNDL
End End;

Function ConSetOptions(Handle: Integer; szValue: PChar): Integer; StdCall;
Begin Try
  Assert(Trace('ConSetOptions %d=%s', [Handle, szValue]));
  TBdpConnection(Connections[Handle]).ParseOptions(szValue);
  Result:= SQL_SUCCESS;
Except
  Result:= DBXERR_INVALIDHNDL
End End;

Function ConTransact(Handle: Integer; Mode: Integer; ilTransID: Integer): Integer; StdCall;
Var
  Connection: TBdpConnection;
  Trans: TTransactionDesc;
Begin Try
  Connection:= TBdpConnection(Connections[Handle]);
  Trans.TransactionID:= ilTransID;
  Case Mode Of
    1: Result:= Connection.beginTransaction(LongWord(@Trans));
    2: Result:= Connection.Commit(LongWord(@Trans));
    4: Result:= Connection.Rollback(LongWord(@Trans));
  Else
    Result:= DBXERR_INVALIDXTXNID
  End
Except
  Result:= DBXERR_INVALIDHNDL
End End;

Function ConGetConnError(Handle: Integer; Error: PChar; Len: Integer): Integer; StdCall;
Var
  TmpMsg: PChar;
  MessageLen: SmallInt;
  Connection: TBdpConnection;
Begin Try
  Connection:= TBdpConnection(Connections[Handle]);
  Error[0]:= #0;
  Result:= Connection.getErrorMessageLen(MessageLen);
  If (Result = SQL_SUCCESS) And (MessageLen > 0) Then Begin
    TmpMsg:= AllocMem(MessageLen + 1);
    Result:= Connection.getErrorMessage(TmpMsg);
    StrLCopy(Error, TmpMsg, Len);
    FreeMem(TmpMsg)
  End
Except
  Result:= DBXERR_INVALIDHNDL
End End;

Function ConGetCommand(Handle: Integer; Var Command: Integer): Integer; StdCall;
Begin Try
  Command:= Commands.Add(TBdpCommand.Create(TBdpConnection(Connections[Handle])));
  Assert(Trace('ConGetCommand %d -> %d', [Handle, Command]));
  Result:= SQL_SUCCESS
Except
  Result:= DBXERR_INVALIDHNDL
End End;

Function ConGetMetadata(Handle: Integer; Var MetaHandle: Integer): Integer; StdCall;
Begin Try
  MetaHandle:= MetaData.Add(TBdpMetadata.Create(TBdpConnection(Connections[Handle])));
  Assert(Trace('ConGetMetadata %d -> %d', [Handle, MetaHandle]));
  Result:= SQL_SUCCESS
Except
  Result:= DBXERR_INVALIDHNDL
End End;

{ BdpCommand related }

Function CmdSetOption (Handle: Integer; Option: Integer; ulValue: Integer): Integer; StdCall;
Var
  CmdOption: TSQLCommandOption;
Begin Try
  Case Option Of
    19: CmdOption:= eCommStoredProc;
  Else
    Result:= DBXERR_NOTSUPPORTED;
    Exit
  End;
  Result:= TBdpCommand(Commands[Handle]).SetOption(CmdOption, ulValue);
Except
  Result:= DBXERR_INVALIDHNDL
End End;

Function CmdPrepare (Handle: Integer; szSQL: PChar; ParamCount: Word; Var ColCount: Integer): Integer; StdCall;
Var
  Command: TBdpCommand;
Begin Try
  Command:= TBdpCommand(Commands[Handle]);
  Result:= Command.Prepare(szSQL, ParamCount);
  If Result=0 Then
    ColCount:= Command.ColumnCount
Except
  Result:= DBXERR_INVALIDHNDL
End End;

Function CmdExecute (Handle: Integer; Var CursorHandle: Integer): Integer; StdCall;
Var
  Cursor: TCA400SQLCursor;
Begin Try
  Result:= TBdpCommand(Commands[Handle]).InternalExecute(Cursor);
  If Result=0 Then
    CursorHandle:= Cursors.Add(Cursor)
  Else
    CursorHandle:= -1
Except
  Result:= DBXERR_INVALIDHNDL
End End;

Function CmdRowsAffected (Handle: Integer; Var RowsAffected: LongWord): Integer; StdCall;
Begin Try
  Result:= TBdpCommand(Commands[Handle]).getRowsAffected(RowsAffected)
Except
  Result:= DBXERR_INVALIDHNDL
End End;

Function CmdGetError(Handle: Integer; Error: PChar; Len: Integer): Integer; StdCall;
Var
  TmpMsg: PChar;
  MessageLen: SmallInt;
  Command: TBdpCommand;
Begin Try
  Command:= TBdpCommand(Commands[Handle]);
  Error[0]:= #0;
  Result:= Command.getErrorMessageLen(MessageLen);
  If (Result = SQL_SUCCESS) And (MessageLen > 0) Then Begin
    TmpMsg:= AllocMem(MessageLen + 1);
    Result:= Command.getErrorMessage(TmpMsg);
    StrLCopy(Error, TmpMsg, Len);
    FreeMem(TmpMsg)
  End
Except
  Result:= DBXERR_INVALIDHNDL
End End;

Function CmdSetParameterInt16 (Handle: Integer; uParameter, uChildPos: Word; eParamDir: TSTMTParamType; uType, subType: Word; iPrecision, iScale: Integer; iLen: LongWord; Data: Word; lInd: Integer): Integer; StdCall;
Begin Try
  Result:= TBdpCommand(Commands[Handle]).SetParameter(uParameter+1, uChildPos, eParamDir, DbxType(uType), DbxSubType(subType), iPrecision, iScale, iLen, @Data, lInd)
Except
  Result:= DBXERR_INVALIDHNDL
End End;

Function CmdSetParameterBool (Handle: Integer; uParameter, uChildPos: Word; eParamDir: TSTMTParamType; uType, subType: Word; iPrecision, iScale: Integer; iLen: LongWord; Data: LongBool; lInd: Integer): Integer; StdCall;
Begin Try
  Result:= TBdpCommand(Commands[Handle]).SetParameter(uParameter+1, uChildPos, eParamDir, DbxType(uType), DbxSubType(subType), iPrecision, iScale, iLen, @Data, lInd)
Except
  Result:= DBXERR_INVALIDHNDL
End End;

Function CmdSetParameterInt32 (Handle: Integer; uParameter, uChildPos: Word; eParamDir: TSTMTParamType; uType, subType: Word; iPrecision, iScale: Integer; iLen: LongWord; Data: Integer; lInd: Integer): Integer; StdCall;
Begin Try
  Result:= TBdpCommand(Commands[Handle]).SetParameter(uParameter+1, uChildPos, eParamDir, DbxType(uType), DbxSubType(subType), iPrecision, iScale, iLen, @Data, lInd)
Except
  Result:= DBXERR_INVALIDHNDL
End End;

Function CmdSetParameterInt64 (Handle: Integer; uParameter, uChildPos: Word; eParamDir: TSTMTParamType; uType, subType: Word; iPrecision, iScale: Integer; iLen: LongWord; Data: Int64; lInd: Integer): Integer; StdCall;
Begin Try
  Result:= TBdpCommand(Commands[Handle]).SetParameter(uParameter+1, uChildPos, eParamDir, DbxType(uType), DbxSubType(subType), iPrecision, iScale, iLen, @Data, lInd)
Except
  Result:= DBXERR_INVALIDHNDL
End End;

Function CmdSetParameterFloat (Handle: Integer; uParameter, uChildPos: Word; eParamDir: TSTMTParamType; uType, subType: Word; iPrecision, iScale: Integer; iLen: LongWord; Data: Double; lInd: Integer): Integer; StdCall;
Begin Try
  Result:= TBdpCommand(Commands[Handle]).SetParameter(uParameter+1, uChildPos, eParamDir, DbxType(uType), DbxSubType(subType), iPrecision, iScale, iLen, @Data, lInd)
Except
  Result:= DBXERR_INVALIDHNDL
End End;

Function CmdSetParameterDouble (Handle: Integer; uParameter, uChildPos: Word; eParamDir: TSTMTParamType; uType, subType: Word; iPrecision, iScale: Integer; iLen: LongWord; Data: Double; lInd: Integer): Integer; StdCall;
Begin Try
  Result:= TBdpCommand(Commands[Handle]).SetParameter(uParameter+1, uChildPos, eParamDir, DbxType(uType), DbxSubType(subType), iPrecision, iScale, iLen, @Data, lInd)
Except
  Result:= DBXERR_INVALIDHNDL
End End;

Function CmdSetParameterString (Handle: Integer; uParameter, uChildPos: Word; eParamDir: TSTMTParamType; uType, subType: Word; iPrecision, iScale: Integer; iLen: LongWord; Data: PChar; lInd: Integer): Integer; StdCall;
Begin Try
  Result:= TBdpCommand(Commands[Handle]).SetParameter(uParameter+1, uChildPos, eParamDir, DbxType(uType), DbxSubType(subType), iPrecision, iScale, iLen, Data, lInd)
Except
  Result:= DBXERR_INVALIDHNDL
End End;

Function CmdSetParameterDecimal (Handle: Integer; uParameter, uChildPos: Word; eParamDir: TSTMTParamType; uType, subType: Word; iPrecision, iScale: Integer; iLen: LongWord; Data: PChar; lInd: Integer): Integer; StdCall;
Var
  ABcd: TBcd;
Begin Try
  ABcd:= StrToBcd(Data);
  Result:= TBdpCommand(Commands[Handle]).SetParameter(uParameter+1, uChildPos, eParamDir, DbxType(uType), DbxSubType(subType), iPrecision, iScale, iLen, @ABcd, lInd)
Except
  Result:= DBXERR_INVALIDHNDL
End End;

Function CmdGetParameterInt16 (Handle: Integer; uParameter, uChildPos: Word; Var Data: Word; lInd: Integer): Integer; StdCall;
Begin Try
  Result:= TBdpCommand(Commands[Handle]).bdpGetParameter(uParameter+1, uChildPos, fldINT16, @Data, SizeOf(Data), lInd)
Except
  Result:= DBXERR_INVALIDHNDL
End End;

Function CmdGetParameterBool (Handle: Integer; uParameter, uChildPos: Word; Var Data: LongBool; lInd: Integer): Integer; StdCall;
Begin Try
  Result:= TBdpCommand(Commands[Handle]).bdpGetParameter(uParameter+1, uChildPos, fldBOOL, @Data, SizeOf(Data), lInd)
Except
  Result:= DBXERR_INVALIDHNDL
End End;

Function CmdGetParameterInt32 (Handle: Integer; uParameter, uChildPos: Word; Var Data: Integer; lInd: Integer): Integer; StdCall;
Begin Try
  Result:= TBdpCommand(Commands[Handle]).bdpGetParameter(uParameter+1, uChildPos, fldINT32, @Data, SizeOf(Data), lInd)
Except
  Result:= DBXERR_INVALIDHNDL
End End;

Function CmdGetParameterInt64 (Handle: Integer; uParameter, uChildPos: Word; Var Data: Int64; lInd: Integer): Integer; StdCall;
Begin Try
  Result:= TBdpCommand(Commands[Handle]).bdpGetParameter(uParameter+1, uChildPos, fldINT64, @Data, SizeOf(Data), lInd)
Except
  Result:= DBXERR_INVALIDHNDL
End End;

Function CmdGetParameterFloat (Handle: Integer; uParameter, uChildPos: Word; Var Data: Double; lInd: Integer): Integer; StdCall;
Begin Try
  Result:= TBdpCommand(Commands[Handle]).bdpGetParameter(uParameter+1, uChildPos, fldFLOAT, @Data, SizeOf(Data), lInd)
Except
  Result:= DBXERR_INVALIDHNDL
End End;

Function CmdGetParameterDouble (Handle: Integer; uParameter, uChildPos: Word; Var Data: Double; lInd: Integer): Integer; StdCall;
Begin Try
  Result:= TBdpCommand(Commands[Handle]).bdpGetParameter(uParameter+1, uChildPos, fldFLOAT, @Data, SizeOf(Data), lInd)
Except
  Result:= DBXERR_INVALIDHNDL
End End;

Function CmdGetParameterString (Handle: Integer; uParameter, uChildPos: Word; Data: PChar; Len: Integer; lInd: Integer): Integer; StdCall;
Begin Try
  Result:= TBdpCommand(Commands[Handle]).bdpGetParameter(uParameter+1, uChildPos, fldZSTRING, Data, Len, lInd)
Except
  Result:= DBXERR_INVALIDHNDL
End End;

Function CmdGetParameterDecimal (Handle: Integer; uParameter, uChildPos: Word; Data: PChar; Len: Integer; lInd: Integer): Integer; StdCall;
Var
  ABcd: TBcd;
Begin Try
  Result:= TBdpCommand(Commands[Handle]).bdpGetParameter(uParameter+1, uChildPos, fldZSTRING, @ABcd, SizeOf(ABcd), lInd);
  If Result=0 Then
    StrLCopy(Data, PChar(BcdToStr(ABcd)), Len)
Except
  Result:= DBXERR_INVALIDHNDL
End End;

{ BdpCursor related }

Function CurNext (Handle: Integer): Integer; StdCall;
Begin Try
  Result:= TBdpCursor(Cursors[Handle]).Next
Except
  Result:= DBXERR_INVALIDHNDL
End End;

Function CurGetColumnName (Handle: Integer; ColumnNumber: Integer; Value: PChar; Len: Integer): Integer; StdCall;
Var
  Cursor: TBdpCursor;
  Res: Word;
Begin Try
  Cursor:= TBdpCursor(Cursors[Handle]);
  If Cursor.checkColumn(Res, ColumnNumber) Then
    StrLCopy(Value, PChar(Cursor.Cols[ColumnNumber-1].Name), Len);
  Result:= Res;
Except
  Result:= DBXERR_INVALIDHNDL
End End;

Function CurGetColumnTypeName (Handle: Integer; ColumnNumber: Integer; Value: PChar; Len: Integer): Integer; StdCall;
Var
  Cursor: TBdpCursor;
  Res: Word;
Begin Try
  Cursor:= TBdpCursor(Cursors[Handle]);
  If Cursor.checkColumn(Res, ColumnNumber) Then
    StrLCopy(Value, PChar(Cursor.Cols[ColumnNumber-1].TypeName), Len);
  Result:= Res;
Except
  Result:= DBXERR_INVALIDHNDL
End End;

Function CurGetColumnType (Handle: Integer; ColumnNumber: Integer; Var uType: Word; Var uSubType: Word): Integer; StdCall;
Begin Try
  Result:= TBdpCursor(Cursors[Handle]).GetColumnType(ColumnNumber, uType, uSubType);
  If Result=0 Then Begin
    uType:= BdpType(uType);
    uSubType:= BdpSubType(uSubType)
  End
Except
  Result:= DBXERR_INVALIDHNDL
End End;

Function CurGetColumnLength (Handle: Integer; ColumnNumber: Integer; Var ulLength: LongWord): Integer; StdCall;
Begin Try
  Result:= TBdpCursor(Cursors[Handle]).GetColumnLength(ColumnNumber, ulLength)
Except
  Result:= DBXERR_INVALIDHNDL
End End;

Function CurGetIsNull (Handle: Integer; ColumnNumber: Integer; Var iInd: LongBool): Integer; StdCall;
Begin Try
  Result:= TBdpCursor(Cursors[Handle]).isNullable(ColumnNumber, iInd)
Except
  Result:= DBXERR_INVALIDHNDL
End End;

Function CurGetInt16 (Handle: Integer; ColumnNumber: Integer; Var Data: Word; Var iInd: LongBool): Integer; StdCall;
Begin Try
  Result:= TBdpCursor(Cursors[Handle]).getShort(ColumnNumber, @Data, iInd)
Except
  Result:= DBXERR_INVALIDHNDL
End End;

Function CurGetInt32 (Handle: Integer; ColumnNumber: Integer; Var Data: Integer; Var iInd: LongBool): Integer; StdCall;
Begin Try
  Result:= TBdpCursor(Cursors[Handle]).getLong(ColumnNumber, @Data, iInd)
Except
  Result:= DBXERR_INVALIDHNDL
End End;

Function CurGetInt64 (Handle: Integer; ColumnNumber: Integer; Var Data: Int64; Var iInd: LongBool): Integer; StdCall;
Begin Try
  Result:= TBdpCursor(Cursors[Handle]).getInt64(ColumnNumber, @Data, iInd)
Except
  Result:= DBXERR_INVALIDHNDL
End End;

Function CurGetDouble (Handle: Integer; ColumnNumber: Integer; Var Data: Double; Var iInd: LongBool): Integer; StdCall;
Begin Try
  Result:= TBdpCursor(Cursors[Handle]).getDouble(ColumnNumber, @Data, iInd)
Except
  Result:= DBXERR_INVALIDHNDL
End End;

Function CurGetFloat (Handle: Integer; ColumnNumber: Integer; Var Data: Double; Var iInd: LongBool): Integer; StdCall;
Begin Try
  Result:= TBdpCursor(Cursors[Handle]).getDouble(ColumnNumber, @Data, iInd)
Except
  Result:= DBXERR_INVALIDHNDL
End End;

Function CurGetString (Handle: Integer; ColumnNumber: Integer; Data: PChar; Var iInd: LongBool): Integer; StdCall;
Begin Try
  Result:= TBdpCursor(Cursors[Handle]).getString(ColumnNumber, Data, iInd)
Except
  Result:= DBXERR_INVALIDHNDL
End End;

Function CurGetDecimalAsString (Handle: Integer; ColumnNumber: Integer; Data: PChar; Var iInd: LongBool): Integer; StdCall;
Var
  ABcd: TBcd;
Begin Try
  Result:= TBdpCursor(Cursors[Handle]).getBcd(ColumnNumber, @ABcd, iInd);
  If Result=0 Then
    StrCopy(Data, PChar(BcdToStr(ABcd)))
Except
  Result:= DBXERR_INVALIDHNDL
End End;

{ BdpMetadata related }

Function MetFree(Handle: Integer): Integer; StdCall;
Begin Try
  TBdpMetadata(MetaData[Handle]).Free;
  MetaData.Remove(Handle);
  Result:= SQL_SUCCESS
Except
  Result:= DBXERR_INVALIDHNDL
End End;

Function MetOpenSchema(Handle: Integer; eMetaDataType: Integer; CatalogName, OwnerName, ObjectName, ColName: PChar; ObjType: Integer): Integer; StdCall;
Var
  Meta: TBdpMetadata;
Begin Try
  Assert(Trace('MetOpenSchema %d, Type=%d, Owner=%s, Object=%s, Col=%s', [Handle, eMetaDataType, OwnerName, ObjectName, ColName]));
  Meta:= TBdpMetadata(MetaData[Handle]);
  Case eMetaDataType Of
    0: Result:= Meta.bdpGetTables(CatalogName, OwnerName, ObjectName, ObjType);
    1: Result:= Meta.bdpGetColumns(CatalogName, OwnerName, ObjectName, ColName, ObjType);
    2: Result:= Meta.bdpGetProcedures(CatalogName, OwnerName, ObjectName, ObjType);
    3: Result:= Meta.bdpGetProcedureParams(CatalogName, OwnerName, ObjectName, ColName);
    4,
    5: Result:= Meta.bdpGetIndices(CatalogName, OwnerName, ObjectName, ObjType);
  Else
    Result:= DBXERR_INVALIDHNDL
  End;
Except
  Result:= DBXERR_INVALIDHNDL
End End;

Function MetCloseSchema(Handle: Integer): Integer; StdCall;
Begin Try
  TBdpMetadata(MetaData[Handle]).MDCursor.Free;
  Result:= SQL_SUCCESS
Except
  Result:= DBXERR_INVALIDHNDL
End End;

Function MetNext(Handle: Integer): Integer; StdCall;
Begin Try
  Result:= TBdpMetadataCursor(TBdpMetadata(MetaData[Handle]).MDCursor).Next
Except
  Result:= DBXERR_EOF;
End End;

Function MetGetInt32 (Handle: Integer; ColumnNumber: Integer; Var Data: Integer; Var iInd: LongBool): Integer; StdCall;
Begin Try
  Result:= TBdpMetadataCursor(TBdpMetadata(MetaData[Handle]).MDCursor).getLong(ColumnNumber, @Data, iInd);
Except
  Result:= DBXERR_INVALIDHNDL
End End;

Function MetGetString (Handle: Integer; ColumnNumber: Integer; Data: PChar; Var iInd: LongBool): Integer; StdCall;
Begin Try
  Result:= TBdpMetadataCursor(TBdpMetadata(MetaData[Handle]).MDCursor).getString(ColumnNumber, Data, iInd);
  If (Result=SQL_SUCCESS) and not (iInd) Then
    Data[Length(Trim(Data))]:= #0;
Except
  Result:= DBXERR_INVALIDHNDL
End End;

Function MetGetType (Handle: Integer; Var uType: Word; Var uSubType: Word; Var iInd: LongBool): Integer; StdCall;
Begin Try
  Result:= TBdpMetadataCursor(TBdpMetadata(MetaData[Handle]).MDCursor).getTypeSubType(uType, uSubType, iInd);
  If (Result=SQL_SUCCESS) and not (iInd) Then Begin
    uType:= BdpType(uType);
    uSubType:= BdpType(uSubType)
  End
Except
  Result:= DBXERR_INVALIDHNDL
End End;

Exports
  ConInit,
  ConExit,
  ConAlloc,
  ConFree,
  ConConnect,
  ConDisConnect,
  ConSetOptions,
  ConTransact,
  ConGetConnError,
  ConGetCommand,
  ConGetMetadata,

  CmdSetOption,
  CmdPrepare,
  CmdExecute,
  CmdRowsAffected,
  CmdGetError,
  CmdSetParameterBool,
  CmdSetParameterInt16,
  CmdSetParameterInt32,
  CmdSetParameterInt64,
  CmdSetParameterFloat,
  CmdSetParameterDouble,
  CmdSetParameterString,
  CmdSetParameterDecimal,
  CmdGetParameterBool,
  CmdGetParameterInt16,
  CmdGetParameterInt32,
  CmdGetParameterInt64,
  CmdGetParameterFloat,
  CmdGetParameterDouble,
  CmdGetParameterString,
  CmdGetParameterDecimal,

  CurNext,
  CurGetColumnName,
  CurGetColumnTypeName,
  CurGetColumnType,
  CurGetColumnLength,
  CurGetIsNull,
  CurGetInt16,
  CurGetInt32,
  CurGetInt64,
  CurGetDouble,
  CurGetFloat,
  CurGetString,
  CurGetDecimalAsString,

  MetFree,
  MetOpenSchema,
  MetCloseSchema,
  MetNext,
  MetGetInt32,
  MetGetString,
  MetGetType;

Begin
  Connections:= THandleContainer.Create;
  Commands:= THandleContainer.Create;
  Cursors:= THandleContainer.Create;
  Metadata:= THandleContainer.Create;
End.


